/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize
#define iMouse vec4(0.0,0.0, 0.0,0.0)

/////////////////////////////////////////////////////////////////////////////////

/*
	"gatorball" by wjbgrafx
	
	Based on "Raymarched Reflections"   Uploaded by Shane in 2015-Nov-17
	https://www.shadertoy.com/view/4dt3zn
	
 	"Gardner Cos Clouds"  Uploaded by fab on 2014-Dec-24
	https://www.shadertoy.com/view/lll3z4

	HG_SDF : GLSL LIBRARY FOR BUILDING SIGNED DISTANCE BOUNDS
	http://mercury.sexy/hg_sdf
	
	"Raymarching Primitives" - Created by inigo quilez - iq/2013
	https://www.shadertoy.com/view/Xds3zN

	"Playing with symmetries - Torus"	   by @paulofalcao	
	http://glslsandbox.com/e#29755.0
	
*/
//==============================================================================

#define PI                      3.1415926535897932384626433832795

#define FAR                     250.0
#define MAX_RAY_STEPS           90
#define MAX_REF_STEPS           50
#define MAX_SHADOW_STEPS        20

#define CAM_DIST				65.0 
#define CAM_POS                 vec3( 0.0, 45.0, -CAM_DIST )
#define CAM_FOV_FACTOR          2.0
#define LOOK_AT                 vec3( 0.0, 0.0, 0.0 )
#define LIGHT_POS               vec3( 0.0, 50.0, -50.0 )
#define LIGHT_ATTEN				0.0005

//------------------------------------------------------------------------------
// Function declarations

vec3 getRayDir( vec3 camPos, vec3 viewDir, vec2 pixelPos ) ;
vec2 trace( vec3 rayOrig, vec3 rayDir );
float traceRef( vec3 rayOrig, vec3 rayDir );
float softShadow( vec3 rayOrig, vec3 lightPos, float k );
vec3 getNormal( in vec3 p );
vec3 doColor( in vec3 sp, in vec3 rayDir, in vec3 surfNorm, in vec2 distID );
vec3 skyColor( vec2 pix );

// Plane with normal n (n is normalized) at some distance from the origin
float fPlane(vec3 p, vec3 n, float distanceFromOrigin);
float fSphere(vec3 p, float r);
float fCapsule(vec3 p, float r, float c); 

// Repeat around the origin by a fixed angle.
float pModPolar(inout vec2 p, float repetitions);
// Mirror in both dimensions and at the diagonal, yielding one eighth of the 
// space; translate by dist before mirroring.
vec2 pMirrorOctant (inout vec2 p, vec2 dist);

float sdEllipsoid( in vec3 p, in vec3 r );

vec2 rot( vec2 p, float r );
vec2 rotsim( vec2 p, float s );

//------------------------------------------------------------------------------

// MAP
// ---

vec2 map( vec3 p )
{  
	vec3 p2 = p;
	  
	// Rotations
	p.yz = rot( p.yz, iGlobalTime * 0.43 );
	p.xz = rot( p.xz, iGlobalTime * 0.37 );
	p.xy = rot( p.xy, iGlobalTime * 0.59 );
	
	// Domain operations
	pModPolar( p.yz, 6.0 );
	pModPolar( p.xz, 6.0 );
	
	pMirrorOctant( p.xy, vec2( 6.0 ) );
	pMirrorOctant( p.xz, vec2( 6.0 ) );

	p.xy = rotsim( p.xy, 12.0 + 11.0 * sin( iGlobalTime * 0.11 ) );
	p.y -= 5.0 + 4.0 * sin( iGlobalTime * 0.31 );
	
	p.yz = rotsim( p.yz, 12.0 + 11.0 * sin( iGlobalTime * 0.17 ) );
	p.z -= 5.0 + 4.0 * sin( iGlobalTime * 0.43 );
	
	// Objects	
	p.yz = rot( p.yz, iGlobalTime * -0.73 );
	p.xz = rot( p.xz, iGlobalTime * -0.87 );
	p.xy = rot( p.xy, iGlobalTime * -0.79 );
	
	float objID = 1.0,
	      d1 = sdEllipsoid( p, vec3( 3.1 + sin( iGlobalTime * 0.51 ),
	                                 3.1 + sin( iGlobalTime * 0.53 ), 
							         3.1 + sin( iGlobalTime * 0.59 ) ) ),
	      d2 = fCapsule( p, 1.5, 2.75 );
	      
	vec2 obj1 = vec2( max( d1, -d2 ), objID );
	
	objID = 2.0;
	vec2 obj2 = vec2( fSphere( p, 1.5 ), objID );
	
	objID = 3.0;
	vec2 obj3 = vec2( fPlane( p2, vec3( 0.0, 1.0, 0.0 ), 30.0 ), objID );
	
	//---------------------------------------------------
	
	vec2 closest = obj1;
	closest = closest.s < obj2.s ? closest : obj2;
	closest = closest.s < obj3.s ? closest : obj3;

	return closest;
}

// end map()

//------------------------------------------------------------------------------

// GET OBJECT COLOR
// ----------------

vec3 getObjectColor( vec3 p, vec2 distID )
{    
    vec3 clr = vec3( 1.0 );
	float objNum = distID.t;
	
	if( objNum == 1.0 )
    {
		clr = vec3( abs( sin( iGlobalTime * 0.13 ) ), 
		            abs( sin( iGlobalTime * 0.19 ) ),
			        abs( sin( iGlobalTime * 0.17 ) ) );
	}
	else if ( objNum == 2.0 )
	{
		clr = vec3( abs( sin( iGlobalTime * 0.23 ) ), 
		            abs( sin( iGlobalTime * 0.17 ) ),
			        abs( sin( iGlobalTime * 0.31 ) ) );
	}
	else if ( objNum == 3.0 )	
	{
		clr = vec3( abs( sin( iGlobalTime * 0.17 ) ), 
		            abs( sin( iGlobalTime * 0.23 ) ),
			        abs( sin( iGlobalTime * 0.29 ) ) );
	}
	    
    return clr;
}

// end getObjectColor()

//------------------------------------------------------------------------------

// MAIN IMAGE
// ----------

void mainImage( out vec4 fragColor, in vec2 fragCoord )
{
	// For Gardner Cos Clouds skyColor()
	vec2 st = fragCoord.xy;
	st.x *= 0.5;
	st.x += sin( iGlobalTime * 0.1 ) * 0.5;
	st.y *= 0.75;

	// Adjust aspect ratio, normalize coords, center origin in x-axis.	
	vec2 uv = ( -iResolution.xy + 2.0 * fragCoord.xy ) / iResolution.y;

    vec3 rayDir = getRayDir( CAM_POS, normalize( LOOK_AT - CAM_POS ), uv );   
    vec3 rayOrig = CAM_POS;   
    vec3 lightPos = LIGHT_POS;
	vec3 sceneColor = vec3( 0.0 );
	vec3 skyClr = skyColor( st );
	   
    // FIRST PASS.
    //------------
    vec2 distID = trace( rayOrig, rayDir );
    float totalDist = distID.s;
    
	if ( totalDist >= FAR )
	{
		sceneColor = skyClr;
	}
	else
	{
	    // Fog based off of distance from the camera. 
	    float fog = smoothstep( FAR * 0.9, 0.0, totalDist ); 
	    
	    // Advancing the ray origin to the new hit point.
	    rayOrig += rayDir * totalDist;
	    
	    // Retrieving the normal at the hit point.
	    vec3 surfNorm = getNormal( rayOrig );
	    
	    // Retrieving the color at the hit point.
	    sceneColor = doColor( rayOrig, rayDir, surfNorm, distID );
	    
	    float k = 24.0;
	    float shadow = softShadow( rayOrig, lightPos, k );
	   
	    // SECOND PASS - REFLECTED RAY
	    //----------------------------
	    rayDir = reflect( rayDir, surfNorm );
	    totalDist = traceRef( rayOrig +  rayDir * 0.01, rayDir );
	    rayOrig += rayDir * totalDist;
	    
	    // Retrieving the normal at the reflected hit point.
	    surfNorm = getNormal( rayOrig );
	    
	    // Coloring the reflected hit point, then adding a portion of it to the 
	    // final scene color. Factor is percent of reflected color to add.
	    sceneColor += doColor( rayOrig, rayDir, surfNorm, distID ) * 0.85;
	    
	    // APPLYING SHADOWS
	    //-----------------
	    sceneColor *= shadow;
	    //sceneColor *= fog;
	    sceneColor = mix( sceneColor, skyClr, 1.0 - fog );
	    
	} // end else totalDist < FAR
	
	fragColor = vec4(clamp(sceneColor, 0.0, 1.0), 1.0);
    
}
//------------------------------------------------------------------------------

// TRACE
// -----

// Standard raymarching routine.
vec2 trace( vec3 rayOrig, vec3 rayDir )
{   
    float totalDist = 0.0;
    vec2 distID = vec2( 0.0 );
    
    for ( int i = 0; i < MAX_RAY_STEPS; i++ )
    {
        distID = map( rayOrig + rayDir * totalDist );
        float dist = distID.s;
        
        if( abs( dist ) < 0.0025 || totalDist > FAR ) 
        {
        	break;
        }
        
        totalDist += dist * 0.75;  // Using more accuracy, in the first pass.
    }
    
    return vec2( totalDist, distID.t );
}

// end trace()

//------------------------------------------------------------------------------

// TRACE REFLECTIONS
// -----------------

// Second pass, which is the first, and only, reflected bounce. Virtually the 
// same as above, but with fewer iterations and less accuracy.

// The reason for a second, virtually identical equation is that raymarching is 
// usually a pretty expensive exercise, so since the reflected ray doesn't 
// require as much detail, you can relax things a bit - in the hope of speeding 
// things up a little.

float traceRef( vec3 rayOrig, vec3 rayDir )
{    
    float totalDist = 0.0;
    
    for ( int i = 0; i < MAX_REF_STEPS; i++ )
    {
        float dist = map( rayOrig + rayDir * totalDist ).s;
        
        if( abs( dist ) < 0.0025 || totalDist > FAR ) 
        {
        	break;
        }
        
        totalDist += dist;
    }
    
    return totalDist;
}

// end traceRef()

//------------------------------------------------------------------------------

// SOFT SHADOW
// -----------

// The value "k" is just a fade-off factor that enables you to control how soft  
// you want the shadows to be. Smaller values give a softer penumbra, and larger
// values give a more hard edged shadow.

float softShadow( vec3 rayOrig, vec3 lightPos, float k )
{
    vec3 rayDir = ( lightPos - rayOrig ); // Unnormalized direction ray.

    float shade = 1.0;
    float dist = 0.01;    
    float end = max( length( rayDir ), 0.001 );
    float stepDist = end / float( MAX_SHADOW_STEPS );
    
    rayDir /= end;

    // Max shadow iterations - More iterations make nicer shadows, but slow 
    // things down. Obviously, the lowest number to give a decent shadow is the 
    // best one to choose. 
    for ( int i = 0; i < MAX_SHADOW_STEPS; i++ )
    {
        float h = map( rayOrig + rayDir * dist ).s;

        //shade = min( shade, k * h / dist );
        // Subtle difference. Thanks to IQ for this tidbit.
        shade = min( shade, smoothstep( 0.0, 1.0, k * h / dist)); 

        // So many options here, and none are perfect: dist += min( h, 0.2 ),etc
        dist += min( h, stepDist * 2.0 ); 
        
        // Early exits from accumulative distance function calls tend to be a 
        // good thing.
        if ( h < 0.001 || dist > end ) 
        {
        	break; 
        }
    }

    // I've added 0.5 to the final shade value, which lightens the shadow a bit. 
    // It's a preference thing. Really dark shadows look too brutal to me.
    return min( max( shade, 0.0 ) + 0.5, 1.0 ); 
}

// end softShadow()

//------------------------------------------------------------------------------

// GET NORMAL
// ----------

// Tetrahedral normal, to save a couple of "map" calls. Courtesy of IQ.

vec3 getNormal( in vec3 p )
{
    // Note the slightly increased sampling distance, to alleviate
    // artifacts due to hit point inaccuracies.
    vec2 e = vec2( 0.005, -0.005 ); 
    return normalize( e.xyy * map( p + e.xyy ).s + 
				      e.yyx * map( p + e.yyx ).s + 
				      e.yxy * map( p + e.yxy ).s + 
				      e.xxx * map( p + e.xxx ).s );
}

// end getNormal()

//------------------------------------------------------------------------------

// DO COLOR
// --------

vec3 doColor( in vec3 sp, in vec3 rayDir, in vec3 surfNorm, in vec2 distID )
                                                               
{    
    // Light direction vector.
    vec3 lDir = LIGHT_POS - sp; 

    // Light to surface distance.
    float lDist = max( length( lDir ), 0.001 ); 

    // Normalizing the light vector.
    lDir /= lDist; 
    
    // Attenuating the light, based on distance.
    //float atten = 1.0 / ( 1.0 + lDist * 0.25 + lDist * lDist * 0.05 );
    float atten = 1.0 / ( lDist * lDist * LIGHT_ATTEN );
    
    // Standard diffuse term.
    float diff = max( dot( surfNorm, lDir ), 0.0 );
    
    // Standard specular term.
    float spec = 
            pow( max( dot( reflect( -lDir, surfNorm ), -rayDir ), 0.0 ), 8.0 );
    
    vec3 objCol = getObjectColor( sp, distID );
    
    // Combining the above terms to produce the final scene color.
    vec3 sceneCol = ( objCol * ( diff + 0.15 ) + vec3( 1.0, 0.6, 0.2 ) *
                                                          spec * 2.0 ) * atten;
   
    return sceneCol;   
}

// end doColor()

//------------------------------------------------------------------------------

// GET RAY DIRECTION
// -----------------

vec3 getRayDir( vec3 camPos, vec3 viewDir, vec2 pixelPos ) 
{
    vec3 camRight = normalize( cross( viewDir, vec3( 0.0, 1.0, 0.0 ) ) );
    vec3 camUp = normalize( cross( camRight, viewDir ) );
    
    return normalize( pixelPos.x * camRight + pixelPos.y * camUp + 
                                                    CAM_FOV_FACTOR * viewDir );
}

// end getRayDir()

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------

// SKY COLOR
// ---------
// https://www.shadertoy.com/view/lll3z4
// Gardner Cos Clouds  Uploaded by fab on 2014-Dec-24
/*
 * Gardner Cos Clouds
 *
 * Translated/adapted from the RenderMan implementation in
 * Texturing & Modeling; a Procedural Approach (3rd ed, p. 50)
 */
 
vec3 skyColor( vec2 pix )
{
	const int nTerms = 10;
	
	float zoom = 1.0,
          cloudDensity = 0.0,
          amplitude = 0.45,
          xphase = 0.9 * iGlobalTime,
          yphase = 0.7,
          xfreq = 2.0 * PI * 0.023,
          yfreq = 2.0 * PI * 0.021,
    
          offset = 0.5,
          xoffset = 37.0,
          yoffzet = 523.0,
    
          x = pix.x,
          y = pix.y,
	      scale = 1.0 / iResolution.x * 60.0 * 1.0 / zoom;

    x = x * scale + offset + iGlobalTime * 1.5;
    y = y * scale + offset - iGlobalTime / 2.3;
    
    for ( int i = 0; i < nTerms; i++ )
    {
        float fx = amplitude * ( offset + cos( xfreq * ( x + xphase ) ) );
        float fy = amplitude * ( offset + cos( yfreq * ( y + yphase ) ) );
        cloudDensity += fx * fy;
        xphase = PI * 0.5 * 0.9 * cos( yfreq * y );
        yphase = PI * 0.5 * 1.1 * cos( xfreq * x );
        amplitude *= 0.602;
        xfreq *= 1.9 + float( i ) * .01;
        yfreq *= 2.2 - float( i ) * 0.08;
    }

    return mix( vec3(0.4, 0.6, 0.9 ), vec3( 1.0 ), cloudDensity );   
}

// end skyColor()

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
// Excerpts from HG_SDF : GLSL LIBRARY FOR BUILDING SIGNED DISTANCE BOUNDS
// http://mercury.sexy/hg_sdf 
// Brought to you by MERCURY http://mercury.sexy
// Released as Creative Commons Attribution-NonCommercial (CC BY-NC)

#define PHI (1.618033988749895)
     // PHI (sqrt(5)*0.5 + 0.5)

float fSphere(vec3 p, float r) {
	return length(p) - r;
}

// Plane with normal n (n is normalized) at some distance from the origin
float fPlane(vec3 p, vec3 n, float distanceFromOrigin) {
	return dot(p, n) + distanceFromOrigin;
}

// Capsule: A Cylinder with round caps on both sides
float fCapsule(vec3 p, float r, float c) {
	return mix(length(p.xz) - r, length(vec3(p.x, abs(p.y) - c, p.z)) - r, 
	                                                        step(c, abs(p.y)));
}

// Repeat around the origin by a fixed angle.
// For easier use, num of repetitions is use to specify the angle.
float pModPolar(inout vec2 p, float repetitions) {
	float angle = 2.*PI/repetitions;
	float a = atan(p.y, p.x) + angle/2.;
	float r = length(p);
	float c = floor(a/angle);
	a = mod(a,angle) - angle/2.;
	p = vec2(cos(a), sin(a))*r;
	// For an odd number of repetitions, fix cell index of the cell in -x 
	// direction (cell index would be e.g. -5 and 5 in the two halves of the 
	// cell):
	if (abs(c) >= (repetitions/2.)) c = abs(c);
	return c;
}

// Mirror at an axis-aligned plane which is at a specified distance <dist> 
// from the origin.
float pMirror (inout float p, float dist) {
	float s = sign(p);
	p = abs(p)-dist;
	return s;
}

// Mirror in both dimensions and at the diagonal, yielding one eighth of the 
// space; translate by dist before mirroring.
vec2 pMirrorOctant (inout vec2 p, vec2 dist) {
	vec2 s = sign(p);
	pMirror(p.x, dist.x);
	pMirror(p.y, dist.y);
	if (p.y > p.x)
		p.xy = p.yx;
	return s;
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------

// From "Raymarching Primitives" - // Created by inigo quilez - iq/2013
// https://www.shadertoy.com/view/Xds3zN

float sdEllipsoid( in vec3 p, in vec3 r )
{
    return (length( p/r ) - 1.0) * min(min(r.x,r.y),r.z);
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------

// From Playing with symmetries - Torus	   by @paulofalcao	
// http://glslsandbox.com/e#29755.0

// Rotation around z-axis when vec2 p.xy;
// Rotation around y-axis when vec2 p.xz;
// Rotation around x-axis when vec2 p.yz.
vec2 rot(vec2 p,float r)
{
   vec2 ret;
   ret.x=p.x*cos(r)-p.y*sin(r);
   ret.y=p.x*sin(r)+p.y*cos(r);
   return ret;
}

// When vec2 p.xy, rotational symmetry about z-axis;
// when vec2 p.xz, rotational symmetry about y-axis
// when vec2 p.yz, rotational symmetry about x-axis
vec2 rotsim(vec2 p,float s)
{
   vec2 ret=p;
   ret=rot(p,-PI/(s*2.0));
   ret=rot(p,floor(atan(ret.x,ret.y)/PI*s)*(PI/s));
   return ret;
}

//------------------------------------------------------------------------------

void main ( void )
{
   mainImage ( gl_FragColor, gl_FragCoord.xy );
}
